# This code example is created for educational purpose
# by Thorsten Thormaehlen (contact: www.thormae.de).
# It is distributed without any warranty.

# Requires PyOpenGL, which can be install with:
# pip install PyOpenGL PyOpenGL_accelerate numpy

from OpenGL.GL import *
from OpenGL.GLUT import *
import numpy as np
import sys
import ctypes

class Renderer:
    def __init__(self):
        self.triangleVertNo = 0
        self.progID = 0
        self.vertID = 0
        self.fragID = 0
        self.vertexLoc = -1
        self.colorLoc = -1
        self.vaoID = 0
        self.bufID = 0

    # public member functions
    def init(self):
        glEnable(GL_DEPTH_TEST)

        self.setup_shaders()

        # create a Vertex Array Objects (VAO)
        self.vaoID = glGenVertexArrays(1)
        
        # generate a Vertex Buffer Object (VBO)
        self.bufID = glGenBuffers(1)

        # binding the Triangle VAO
        glBindVertexArray(self.vaoID)

        triangle_vertex_data = np.array([
             0.0,  0.5, 0.0, 1.0, 0.0, 0.0, 1.0,  
            -0.5, -0.5, 0.0, 0.0, 1.0, 0.0, 1.0,  
             0.5, -0.5, 0.0, 0.0, 0.0, 1.0, 1.0, 
        ], dtype=np.float32)
        self.triangleVertNo = 3

        glBindBuffer(GL_ARRAY_BUFFER, self.bufID)
        glBufferData(GL_ARRAY_BUFFER, triangle_vertex_data.nbytes, triangle_vertex_data, GL_STATIC_DRAW)

        # 7 floats * 4 bytes per float = 28 bytes
        stride = 7 * 4 
        offset = 0

        # position
        if self.vertexLoc != -1:
            glVertexAttribPointer(self.vertexLoc, 3, GL_FLOAT, GL_FALSE, stride, ctypes.c_void_p(offset))
            glEnableVertexAttribArray(self.vertexLoc)

        # color
        if self.colorLoc != -1:
            offset = 3 * 4
            glVertexAttribPointer(self.colorLoc, 4, GL_FLOAT, GL_FALSE, stride, ctypes.c_void_p(offset))
            glEnableVertexAttribArray(self.colorLoc)

    def setup_shaders(self):

        # create shaders
        self.vertID = glCreateShader(GL_VERTEX_SHADER)
        self.fragID = glCreateShader(GL_FRAGMENT_SHADER)

        # load shader source from file
        vs_source = self._load_shader_src("first.vert")
        fs_source = self._load_shader_src("first.frag")

        # specify shader source
        glShaderSource(self.vertID, vs_source)
        glShaderSource(self.fragID, fs_source)

        # compile shaders
        glCompileShader(self.vertID)
        glCompileShader(self.fragID)

        # check for errors
        self._print_shader_info_log(self.vertID)
        self._print_shader_info_log(self.fragID)

        # create program and attach shaders
        self.progID = glCreateProgram()
        glAttachShader(self.progID, self.vertID)
        glAttachShader(self.progID, self.fragID)

        # "outColor" is a user-provided OUT variable
        # of the fragment shader.
        # Its output is bound to the first color buffer
        # in the framebuffer
        glBindFragDataLocation(self.progID, 0, "outputColor")

        # link the program
        glLinkProgram(self.progID)
        # output error messages
        self._print_program_info_log(self.progID)

        # "inputPosition" and "inputColor" are user-provided
        # IN variables of the vertex shader.
        # Their locations are stored to be used later with
        # glEnableVertexAttribArray()
        self.vertexLoc = glGetAttribLocation(self.progID, "inputPosition")
        self.colorLoc = glGetAttribLocation(self.progID, "inputColor")

    def resize(self, w, h):
        glViewport(0, 0, w, h)

    def display(self):
        glClearColor(0.0, 0.0, 0.0, 0.0)
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)

        glUseProgram(self.progID)

        # bind Triangle VAO
        glBindVertexArray(self.vaoID)
        # render data
        glDrawArrays(GL_TRIANGLES, 0, self.triangleVertNo)

    def dispose(self):
        glDeleteVertexArrays(1, [self.vaoID])
        glDeleteBuffers(1, [self.bufID])
        glDeleteProgram(self.progID)
        glDeleteShader(self.vertID)
        glDeleteShader(self.fragID)

    # private member functions
    def _print_shader_info_log(self, shader_obj):
        log = glGetShaderInfoLog(shader_obj)
        if log:
            print(f"Shader Log: {log.decode()}")

    def _print_program_info_log(self, prog_obj):
        log = glGetProgramInfoLog(prog_obj)
        if log:
            print(f"Program Log: {log.decode()}")

    def _load_shader_src(self, filename):
        try:
            with open(filename, 'r') as f:
                return f.read()
        except IOError:
            print(f"Unable to open file {filename}")
            sys.exit(1)

# --- Global Renderer Instance ---
renderer = Renderer()

# --- GLUT Callbacks ---
def glut_resize(w, h):
    renderer.resize(300, 300)

def glut_display():
    renderer.display()
    glutSwapBuffers()
    
def glut_close():
    renderer.dispose()

def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA)
    glutInitWindowPosition(100, 100)
    glutInitWindowSize(320, 320)
    
    glutCreateWindow(b"FirstTriangle")

    # Register Callbacks
    glutDisplayFunc(glut_display)
    glutReshapeFunc(glut_resize)
    glutCloseFunc(glut_close)
    
    # Initialize Renderer
    renderer.init()

    # Enter the main event loop
    glutMainLoop()

if __name__ == "__main__":
    main()